<?php
session_start();

// Check if logged in
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header('Location: index.php');
    exit;
}
?>
<!DOCTYPE html>
<html lang="bn">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard - Admin Panel</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: #f5f7fa;
        }
        
        .header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 20px 30px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        
        .header h1 {
            font-size: 24px;
        }
        
        .header .user-info {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .btn-logout {
            padding: 8px 20px;
            background: rgba(255,255,255,0.2);
            color: white;
            border: 1px solid white;
            border-radius: 5px;
            cursor: pointer;
            text-decoration: none;
            font-size: 14px;
        }
        
        .btn-logout:hover {
            background: rgba(255,255,255,0.3);
        }
        
        .container {
            padding: 30px;
            max-width: 1400px;
            margin: 0 auto;
        }
        
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .stat-card {
            background: white;
            padding: 25px;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.08);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .stat-card .icon {
            font-size: 40px;
            opacity: 0.8;
        }
        
        .stat-card .info h3 {
            font-size: 32px;
            color: #333;
            margin-bottom: 5px;
        }
        
        .stat-card .info p {
            color: #666;
            font-size: 14px;
        }
        
        .tabs {
            display: flex;
            gap: 10px;
            margin-bottom: 20px;
            border-bottom: 2px solid #e0e0e0;
        }
        
        .tab-btn {
            padding: 12px 25px;
            background: transparent;
            border: none;
            cursor: pointer;
            font-size: 16px;
            color: #666;
            border-bottom: 3px solid transparent;
            transition: all 0.3s;
        }
        
        .tab-btn.active {
            color: #667eea;
            border-bottom-color: #667eea;
        }
        
        .tab-content {
            display: none;
        }
        
        .tab-content.active {
            display: block;
        }
        
        .data-card {
            background: white;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.08);
            overflow: hidden;
        }
        
        .data-card-header {
            padding: 20px;
            background: #f8f9fa;
            border-bottom: 1px solid #e0e0e0;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .data-card-header h2 {
            color: #333;
            font-size: 20px;
        }
        
        .refresh-btn {
            padding: 8px 16px;
            background: #667eea;
            color: white;
            border: none;
            border-radius: 5px;
            cursor: pointer;
            font-size: 14px;
        }
        
        .refresh-btn:hover {
            background: #5568d3;
        }
        
        .data-table {
            width: 100%;
            border-collapse: collapse;
        }
        
        .data-table thead {
            background: #f8f9fa;
        }
        
        .data-table th,
        .data-table td {
            padding: 15px;
            text-align: left;
            border-bottom: 1px solid #e0e0e0;
        }
        
        .data-table th {
            font-weight: 600;
            color: #333;
            font-size: 14px;
        }
        
        .data-table td {
            color: #666;
            font-size: 14px;
        }
        
        .data-table tbody tr:hover {
            background: #f8f9fa;
        }
        
        .badge {
            padding: 4px 10px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: 600;
        }
        
        .badge-incoming {
            background: #e3f2fd;
            color: #1976d2;
        }
        
        .badge-outgoing {
            background: #f3e5f5;
            color: #7b1fa2;
        }
        
        .badge-ringing {
            background: #fff3e0;
            color: #f57c00;
        }
        
        .badge-answered {
            background: #e8f5e9;
            color: #388e3c;
        }
        
        .badge-ended {
            background: #fce4ec;
            color: #c2185b;
        }
        
        .loading {
            text-align: center;
            padding: 40px;
            color: #666;
        }
        
        .no-data {
            text-align: center;
            padding: 40px;
            color: #999;
        }
        
        .live-indicator {
            display: inline-block;
            width: 8px;
            height: 8px;
            background: #4caf50;
            border-radius: 50%;
            margin-right: 5px;
            animation: pulse 2s infinite;
        }
        
        @keyframes pulse {
            0%, 100% { opacity: 1; }
            50% { opacity: 0.3; }
        }
        
        .device-selector {
            margin-bottom: 20px;
            display: flex;
            gap: 10px;
            align-items: center;
        }
        
        .device-selector select {
            padding: 10px 15px;
            border: 1px solid #ddd;
            border-radius: 5px;
            font-size: 14px;
            min-width: 250px;
        }
    </style>
</head>
<body>
    <div class="header">
        <h1>🛡️ Admin Dashboard</h1>
        <div class="user-info">
            <span><span class="live-indicator"></span>Live Monitoring</span>
            <a href="logout.php" class="btn-logout">Logout</a>
        </div>
    </div>
    
    <div class="container">
        <div class="stats-grid">
            <div class="stat-card">
                <div class="info">
                    <h3 id="totalDevices">0</h3>
                    <p>মোট ডিভাইস</p>
                </div>
                <div class="icon">📱</div>
            </div>
            
            <div class="stat-card">
                <div class="info">
                    <h3 id="totalSms">0</h3>
                    <p>মোট এসএমএস</p>
                </div>
                <div class="icon">💬</div>
            </div>
            
            <div class="stat-card">
                <div class="info">
                    <h3 id="totalCalls">0</h3>
                    <p>মোট কল</p>
                </div>
                <div class="icon">📞</div>
            </div>
            
            <div class="stat-card">
                <div class="info">
                    <h3 id="lastUpdate">--:--</h3>
                    <p>শেষ আপডেট</p>
                </div>
                <div class="icon">🕒</div>
            </div>
        </div>
        
        <div class="device-selector">
            <label>ডিভাইস নির্বাচন করুন:</label>
            <select id="deviceSelect">
                <option value="">সব ডিভাইস</option>
            </select>
            <button class="refresh-btn" onclick="loadAllData()">🔄 রিফ্রেশ</button>
        </div>
        
        <div class="tabs">
            <button class="tab-btn active" data-tab="devices">ডিভাইস সমূহ</button>
            <button class="tab-btn" data-tab="sms">এসএমএস</button>
            <button class="tab-btn" data-tab="calls">কল লগ</button>
        </div>
        
        <div id="devices" class="tab-content active">
            <div class="data-card">
                <div class="data-card-header">
                    <h2>রেজিস্টার্ড ডিভাইস সমূহ</h2>
                </div>
                <div id="devicesTable"></div>
            </div>
        </div>
        
        <div id="sms" class="tab-content">
            <div class="data-card">
                <div class="data-card-header">
                    <h2>এসএমএস মেসেজ</h2>
                </div>
                <div id="smsTable"></div>
            </div>
        </div>
        
        <div id="calls" class="tab-content">
            <div class="data-card">
                <div class="data-card-header">
                    <h2>কল লগ</h2>
                </div>
                <div id="callsTable"></div>
            </div>
        </div>
    </div>
    
    <script>
        let currentTab = 'devices';
        let selectedDevice = '';
        let autoRefreshInterval;
        
        // Tab switching
        document.querySelectorAll('.tab-btn').forEach(btn => {
            btn.addEventListener('click', function() {
                document.querySelectorAll('.tab-btn').forEach(b => b.classList.remove('active'));
                document.querySelectorAll('.tab-content').forEach(c => c.classList.remove('active'));
                
                this.classList.add('active');
                currentTab = this.dataset.tab;
                document.getElementById(currentTab).classList.add('active');
            });
        });
        
        // Device selector
        document.getElementById('deviceSelect').addEventListener('change', function() {
            selectedDevice = this.value;
            loadAllData();
        });
        
        // Load all data
        function loadAllData() {
            const url = `../api/get_data.php?type=all&device_id=${selectedDevice}`;
            
            fetch(url)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        updateStats(data.data);
                        updateDevicesList(data.data.devices);
                        updateSmsList(data.data.sms);
                        updateCallsList(data.data.calls);
                        updateLastUpdate();
                    }
                })
                .catch(error => console.error('Error:', error));
        }
        
        // Update statistics
        function updateStats(data) {
            document.getElementById('totalDevices').textContent = data.devices ? data.devices.length : 0;
            document.getElementById('totalSms').textContent = data.sms ? data.sms.length : 0;
            document.getElementById('totalCalls').textContent = data.calls ? data.calls.length : 0;
        }
        
        // Update last update time
        function updateLastUpdate() {
            const now = new Date();
            const timeString = now.toLocaleTimeString('bn-BD', { 
                hour: '2-digit', 
                minute: '2-digit' 
            });
            document.getElementById('lastUpdate').textContent = timeString;
        }
        
        // Update devices list
        function updateDevicesList(devices) {
            const container = document.getElementById('devicesTable');
            
            if (!devices || devices.length === 0) {
                container.innerHTML = '<div class="no-data">কোন ডিভাইস নেই</div>';
                return;
            }
            
            // Update device selector
            const select = document.getElementById('deviceSelect');
            select.innerHTML = '<option value="">সব ডিভাইস</option>';
            devices.forEach(device => {
                const option = document.createElement('option');
                option.value = device.device_id;
                option.textContent = `${device.child_name} (${device.device_id})`;
                select.appendChild(option);
            });
            
            let html = '<table class="data-table"><thead><tr>';
            html += '<th>শিশুর নাম</th>';
            html += '<th>অভিভাবকের নাম</th>';
            html += '<th>ইমেইল</th>';
            html += '<th>ডিভাইস আইডি</th>';
            html += '<th>রেজিস্ট্রেশন সময়</th>';
            html += '</tr></thead><tbody>';
            
            devices.forEach(device => {
                html += '<tr>';
                html += `<td>${device.child_name}</td>`;
                html += `<td>${device.parent_name}</td>`;
                html += `<td>${device.parent_email}</td>`;
                html += `<td>${device.device_id}</td>`;
                html += `<td>${new Date(device.created_at).toLocaleString('bn-BD')}</td>`;
                html += '</tr>';
            });
            
            html += '</tbody></table>';
            container.innerHTML = html;
        }
        
        // Update SMS list
        function updateSmsList(smsList) {
            const container = document.getElementById('smsTable');
            
            if (!smsList || smsList.length === 0) {
                container.innerHTML = '<div class="no-data">কোন এসএমএস নেই</div>';
                return;
            }
            
            let html = '<table class="data-table"><thead><tr>';
            html += '<th>প্রেরক</th>';
            html += '<th>মেসেজ</th>';
            html += '<th>ডিভাইস আইডি</th>';
            html += '<th>সময়</th>';
            html += '</tr></thead><tbody>';
            
            smsList.forEach(sms => {
                html += '<tr>';
                html += `<td><strong>${sms.sender}</strong></td>`;
                html += `<td>${sms.message}</td>`;
                html += `<td style="font-size: 12px; color: #999;">${sms.device_id}</td>`;
                html += `<td>${new Date(sms.received_at).toLocaleString('bn-BD')}</td>`;
                html += '</tr>';
            });
            
            html += '</tbody></table>';
            container.innerHTML = html;
        }
        
        // Update calls list
        function updateCallsList(callsList) {
            const container = document.getElementById('callsTable');
            
            if (!callsList || callsList.length === 0) {
                container.innerHTML = '<div class="no-data">কোন কল লগ নেই</div>';
                return;
            }
            
            let html = '<table class="data-table"><thead><tr>';
            html += '<th>ফোন নম্বর</th>';
            html += '<th>কল টাইপ</th>';
            html += '<th>স্ট্যাটাস</th>';
            html += '<th>সময়কাল</th>';
            html += '<th>ডিভাইস আইডি</th>';
            html += '<th>সময়</th>';
            html += '</tr></thead><tbody>';
            
            callsList.forEach(call => {
                html += '<tr>';
                html += `<td><strong>${call.phone_number}</strong></td>`;
                html += `<td><span class="badge badge-${call.call_type}">${call.call_type}</span></td>`;
                html += `<td><span class="badge badge-${call.call_status}">${call.call_status}</span></td>`;
                html += `<td>${call.duration} সেকেন্ড</td>`;
                html += `<td style="font-size: 12px; color: #999;">${call.device_id}</td>`;
                html += `<td>${new Date(call.call_time).toLocaleString('bn-BD')}</td>`;
                html += '</tr>';
            });
            
            html += '</tbody></table>';
            container.innerHTML = html;
        }
        
        // Initial load
        loadAllData();
        
        // Auto-refresh every 5 seconds
        autoRefreshInterval = setInterval(loadAllData, 5000);
        
        // Clear interval on page unload
        window.addEventListener('beforeunload', function() {
            clearInterval(autoRefreshInterval);
        });
    </script>
</body>
</html>
