<?php
session_start();
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header('Location: index.php');
    exit();
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Application Management - Admin Panel</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; background: #f5f7fa; }
        .header { background: linear-gradient(135deg, #10b981 0%, #059669 100%); color: white; padding: 20px 40px; display: flex; justify-content: space-between; align-items: center; }
        .back-btn { background: rgba(255,255,255,0.2); color: white; padding: 10px 20px; border: none; border-radius: 5px; cursor: pointer; text-decoration: none; }
        .container { max-width: 1400px; margin: 30px auto; padding: 0 20px; }
        .filters { display: flex; gap: 10px; margin-bottom: 20px; }
        .filter-btn { padding: 10px 20px; border: 2px solid #e5e7eb; background: white; border-radius: 6px; cursor: pointer; transition: all 0.3s; }
        .filter-btn.active { background: #10b981; color: white; border-color: #10b981; }
        .content { background: white; border-radius: 10px; padding: 20px; box-shadow: 0 2px 10px rgba(0,0,0,0.05); }
        table { width: 100%; border-collapse: collapse; }
        th { background: #f9fafb; padding: 12px; text-align: left; font-weight: 600; border-bottom: 2px solid #e5e7eb; }
        td { padding: 12px; border-bottom: 1px solid #e5e7eb; }
        .status { padding: 5px 12px; border-radius: 20px; font-size: 12px; font-weight: bold; }
        .status.pending { background: #fef3c7; color: #92400e; }
        .status.approved { background: #d1fae5; color: #065f46; }
        .status.rejected { background: #fee2e2; color: #991b1b; }
        .btn { padding: 6px 12px; border: none; border-radius: 5px; cursor: pointer; font-size: 12px; margin: 2px; }
        .btn-view { background: #3b82f6; color: white; }
        .btn-approve { background: #10b981; color: white; }
        .btn-reject { background: #ef4444; color: white; }
        .modal { display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 1000; }
        .modal-content { background: white; width: 90%; max-width: 800px; margin: 50px auto; border-radius: 10px; padding: 30px; max-height: 80vh; overflow-y: auto; }
        .modal-close { float: right; font-size: 28px; cursor: pointer; }
        .detail-row { display: grid; grid-template-columns: 200px 1fr; gap: 10px; padding: 10px; border-bottom: 1px solid #e5e7eb; }
        .detail-label { font-weight: 600; color: #374151; }
        .photo-preview { max-width: 200px; border-radius: 10px; }
    </style>
</head>
<body>
    <div class="header">
        <h1>📋 Application Management</h1>
        <a href="dashboard.php" class="back-btn">← Back to Dashboard</a>
    </div>
    
    <div class="container">
        <div class="filters">
            <button class="filter-btn active" onclick="filterStatus('')">All Applications</button>
            <button class="filter-btn" onclick="filterStatus('pending')">Pending</button>
            <button class="filter-btn" onclick="filterStatus('approved')">Approved</button>
            <button class="filter-btn" onclick="filterStatus('rejected')">Rejected</button>
        </div>
        
        <div class="content">
            <table id="applicationsTable">
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Name</th>
                        <th>Email</th>
                        <th>Phone</th>
                        <th>Nationality</th>
                        <th>Submitted</th>
                        <th>Status</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody id="tableBody"></tbody>
            </table>
        </div>
    </div>
    
    <div id="detailModal" class="modal">
        <div class="modal-content">
            <span class="modal-close" onclick="closeModal()">&times;</span>
            <h2>Application Details</h2>
            <div id="detailContent"></div>
        </div>
    </div>
    
    <script>
        let currentFilter = '';
        
        async function loadApplications() {
            const response = await fetch(`../api/get_applications.php?status=${currentFilter}`);
            const data = await response.json();
            renderTable(data.applications || []);
        }
        
        function renderTable(apps) {
            const tbody = document.getElementById('tableBody');
            if (apps.length === 0) {
                tbody.innerHTML = '<tr><td colspan="8" style="text-align:center;padding:40px;color:#9ca3af;">No applications found</td></tr>';
                return;
            }
            
            tbody.innerHTML = apps.map(app => `
                <tr>
                    <td>${app.id}</td>
                    <td>${app.given_name} ${app.family_name}</td>
                    <td>${app.email}</td>
                    <td>${app.phone}</td>
                    <td>${app.nationality || 'N/A'}</td>
                    <td>${app.submitted_at}</td>
                    <td><span class="status ${app.status}">${app.status.toUpperCase()}</span></td>
                    <td>
                        <button class="btn btn-view" onclick="viewDetails(${app.id})">View</button>
                        ${app.status === 'pending' ? `
                            <button class="btn btn-approve" onclick="approveApp(${app.id})">Approve</button>
                            <button class="btn btn-reject" onclick="rejectApp(${app.id})">Reject</button>
                        ` : ''}
                    </td>
                </tr>
            `).join('');
        }
        
        function filterStatus(status) {
            currentFilter = status;
            document.querySelectorAll('.filter-btn').forEach(btn => btn.classList.remove('active'));
            event.target.classList.add('active');
            loadApplications();
        }
        
        async function viewDetails(id) {
            const response = await fetch(`../api/get_applications.php`);
            const data = await response.json();
            const app = data.applications.find(a => a.id === id);
            
            if (!app) return;
            
            const html = `
                <div class="detail-row"><div class="detail-label">Given Name:</div><div>${app.given_name}</div></div>
                <div class="detail-row"><div class="detail-label">Family Name:</div><div>${app.family_name}</div></div>
                <div class="detail-row"><div class="detail-label">Date of Birth:</div><div>${app.date_of_birth}</div></div>
                <div class="detail-row"><div class="detail-label">Place of Birth:</div><div>${app.place_of_birth || 'N/A'}</div></div>
                <div class="detail-row"><div class="detail-label">Sex:</div><div>${app.sex || 'N/A'}</div></div>
                <div class="detail-row"><div class="detail-label">Nationality:</div><div>${app.nationality || 'N/A'}</div></div>
                <div class="detail-row"><div class="detail-label">Email:</div><div>${app.email}</div></div>
                <div class="detail-row"><div class="detail-label">Phone:</div><div>${app.phone}</div></div>
                <div class="detail-row"><div class="detail-label">Address:</div><div>${app.address || 'N/A'}</div></div>
                <div class="detail-row"><div class="detail-label">Passport:</div><div>${app.passport_number || 'N/A'}</div></div>
                ${app.photo_path ? `<div class="detail-row"><div class="detail-label">Photo:</div><div><img src="../${app.photo_path}" class="photo-preview"></div></div>` : ''}
                ${app.signature_path ? `<div class="detail-row"><div class="detail-label">Signature:</div><div><img src="../${app.signature_path}" class="photo-preview"></div></div>` : ''}
                <div class="detail-row"><div class="detail-label">Status:</div><div><span class="status ${app.status}">${app.status.toUpperCase()}</span></div></div>
            `;
            
            document.getElementById('detailContent').innerHTML = html;
            document.getElementById('detailModal').style.display = 'block';
        }
        
        function closeModal() {
            document.getElementById('detailModal').style.display = 'none';
        }
        
        async function approveApp(id) {
            if (!confirm('Approve this application?')) return;
            
            const response = await fetch('../api/application_action.php', {
                method: 'POST',
                headers: {'Content-Type': 'application/json'},
                body: JSON.stringify({application_id: id, action: 'approve'})
            });
            
            const data = await response.json();
            alert(data.message);
            loadApplications();
        }
        
        async function rejectApp(id) {
            if (!confirm('Reject this application?')) return;
            
            const response = await fetch('../api/application_action.php', {
                method: 'POST',
                headers: {'Content-Type': 'application/json'},
                body: JSON.stringify({application_id: id, action: 'reject'})
            });
            
            const data = await response.json();
            alert(data.message);
            loadApplications();
        }
        
        loadApplications();
    </script>
</body>
</html>
